<?php
/**
 * Plugin Name: Secure Login Shield PRO
 * Plugin URI: https://plugins.bentreder.com/secure-login-shield-pro/
 * Description: Hide wp-login.php, set secret login URLs, block bots, and add stealth 404 protection. Advanced login security for WordPress.
 * Version: 1.0
 * Author: Ben Treder
 * Author URI: https://BenTreder.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: secure-login-shield-pro
 * Domain Path: /languages
 */

if ( ! defined('ABSPATH') ) exit;

define('SLS_PRO_VER', '0.9.2');
define('SLS_PRO_OPT', 'sls_pro_settings');
define('SLS_PRO_TABLE', 'sls_pro_logs');
define('SLS_PRO_NONCE', 'sls_pro_nonce');

class SLS_Pro {
    private static $instance = null;
    private $opts;

    public static function instance(){
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct(){
        $this->opts = wp_parse_args(get_option(SLS_PRO_OPT, []), $this->defaults());

        register_activation_hook(__FILE__, [$this,'activate']);
        register_deactivation_hook(__FILE__, [$this,'deactivate']);

        add_action('init', [$this,'add_rewrite_rules']);
        add_action('init', [$this,'maybe_block_access'], 1);
        add_action('login_init', [$this,'honeypot_login_gate'], 0);

        add_action('admin_menu', [$this,'admin_menu']);
        add_action('admin_init', [$this,'register_settings']);
        add_action('admin_post_sls_pro_export', [$this,'handle_export']);

        add_filter('login_redirect', [$this,'role_based_redirect'], 10, 3);

        add_action('show_user_profile', [$this,'user_profile_field']);
        add_action('edit_user_profile', [$this,'user_profile_field']);
        add_action('personal_options_update', [$this,'save_user_profile_field']);
        add_action('edit_user_profile_update', [$this,'save_user_profile_field']);
    }

    private function defaults(){
        return [
            // Default includes /secure-login on fresh install
            'global_secrets'     => ['secure-login'],
            'role_secrets'       => [],
            // IP controls
            'ip_whitelist'       => '',
            'ip_blacklist'       => '',
            'country_mode'       => 'off',       // off|allow|deny
            'country_list'       => '',
            // Schedule
            'schedule_enabled'   => 0,
            'schedule_days'      => ['mon','tue','wed','thu','fri'],
            'schedule_start'     => '09:00',
            'schedule_end'       => '17:00',
            // Honeypot
            'honeypot_enabled'   => 1,
            'honeypot_block_minutes' => 30,
            // Logging
            'log_enabled'        => 1,
            'log_keep_days'      => 30,
            // Redirects
            'role_redirects'     => [],
        ];
    }

    /* ---------------- Activation / Deactivation ---------------- */

    public function activate(){
        global $wpdb;
        $table = $wpdb->prefix . SLS_PRO_TABLE;
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS `$table` (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            occurred_at DATETIME NOT NULL,
            ip VARBINARY(16) NOT NULL,
            path VARCHAR(255) NOT NULL,
            ua TEXT NULL,
            ref TEXT NULL,
            event VARCHAR(64) NOT NULL,
            meta LONGTEXT NULL,
            PRIMARY KEY (id),
            KEY event_idx (event),
            KEY occurred_idx (occurred_at)
        ) $charset;";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        // Save defaults (includes /secure-login) only if option not present
        add_option(SLS_PRO_OPT, $this->defaults());

        $this->add_rewrite_rules();
        flush_rewrite_rules(false);
    }

    public function deactivate(){
        flush_rewrite_rules(false);
    }

    /* ---------------- Rewrite / Secrets ---------------- */

    public function add_rewrite_rules(){
        $slugs = $this->all_secret_slugs();
        foreach ($slugs as $slug){
            $slug = trim($slug, "/ \t\n\r\0\x0B");
            if (!$slug) continue;
            add_rewrite_rule("^" . preg_quote($slug, '/') . "/?$", 'wp-login.php?sls_gate=1&secret=' . rawurlencode($slug), 'top');
        }
    }

    private function all_secret_slugs(){
        $out = [];
        if (!empty($this->opts['global_secrets']) && is_array($this->opts['global_secrets'])){
            $out = array_merge($out, array_values($this->opts['global_secrets']));
        }
        if (!empty($this->opts['role_secrets']) && is_array($this->opts['role_secrets'])){
            foreach($this->opts['role_secrets'] as $s){
                if ($s) $out[] = $s;
            }
        }
        $users = get_users([ 'fields' => ['ID'] ]);
        foreach ($users as $u){
            $s = get_user_meta($u->ID, 'sls_pro_secret_slug', true);
            if ($s) $out[] = $s;
        }
        return array_unique(array_filter(array_map('sanitize_title', $out)));
    }

    /* ---------------- Access Gates ---------------- */

    public function maybe_block_access(){
        if (wp_doing_cron() || (function_exists('wp_is_json_request') && wp_is_json_request())) return;
        if (is_admin() && !is_user_logged_in()){
            return; // WP will redirect to login; our login_init decides honeypot/allow
        }
    }

    public function honeypot_login_gate(){
        if (isset($_GET['sls_gate']) && $_GET['sls_gate'] == '1'){
            if (!$this->passes_all_access_rules()){
                $this->log_event('access-denied', ['reason'=>'rule-fail','where'=>'via_secret']);
                $this->deny_with_403();
            }
            return; // allow real login
        }

        if (!empty($this->opts['honeypot_enabled'])){
            $this->log_event('honeypot-hit', ['method'=>$_SERVER['REQUEST_METHOD'] ?? 'GET']);
            $ip = $this->client_ip();
            if ($ip){
                $key = 'sls_block_' . $ip;
                set_transient($key, 1, MINUTE_IN_SECONDS * max(1, intval($this->opts['honeypot_block_minutes'])));
            }
            $this->render_fake_login();
            exit;
        }
    }

    private function passes_all_access_rules(){
        $ip = $this->client_ip();
        if ($ip && get_transient('sls_block_'.$ip)) return false;
        if (!$this->ip_policy_allows($ip)) return false;
        if (!empty($this->opts['schedule_enabled']) && !$this->within_schedule()) return false;

        if (($this->opts['country_mode'] ?? 'off') !== 'off'){
            $country = apply_filters('sls_pro_detect_country', null, $ip); // supply ISO-2 from a GeoIP plugin
            $list = array_filter(array_map('trim', explode(',', $this->opts['country_list'] ?? '')));
            if ($country){
                if ($this->opts['country_mode'] === 'allow' && !in_array($country, $list, true)) return false;
                if ($this->opts['country_mode'] === 'deny'  &&  in_array($country, $list, true)) return false;
            }
        }
        return true;
    }

    private function ip_policy_allows($ip){
        if (!$ip) return false;
        $white = $this->lines_to_list($this->opts['ip_whitelist'] ?? '');
        $black = $this->lines_to_list($this->opts['ip_blacklist'] ?? '');

        if (!empty($white)){
            $ok = false;
            foreach($white as $rule) if ($this->ip_in_rule($ip, $rule)) { $ok = true; break; }
            if (!$ok) return false;
        }
        foreach($black as $rule) if ($this->ip_in_rule($ip, $rule)) return false;

        return true;
    }

    private function within_schedule(){
        $tz = wp_timezone();
        $now = new DateTime('now', $tz);
        $day = strtolower($now->format('D')); // mon..sun
        $map = [ 'mon','tue','wed','thu','fri','sat','sun' ];
        $enabled_days = array_intersect($map, (array)($this->opts['schedule_days'] ?? []));
        if (!in_array($day, $enabled_days, true)) return false;

        $start = DateTime::createFromFormat('H:i', $this->opts['schedule_start'] ?? '00:00', $tz);
        $end   = DateTime::createFromFormat('H:i', $this->opts['schedule_end']   ?? '23:59', $tz);
        if (!$start || !$end) return true;
        $cur = DateTime::createFromFormat('H:i', $now->format('H:i'), $tz);
        return ($cur >= $start && $cur <= $end);
    }

    /* ---------------- Honeypot rendering ---------------- */

    private function render_fake_login(){
        status_header(200);
        nocache_headers();
        ?>
        <!doctype html><html>
        <head>
            <meta charset="utf-8">
            <meta name="viewport" content="width=device-width,initial-scale=1">
            <title>Log In ‹ WordPress</title>
            <style>
                :root{
                    /* Neutral light palette: light blues + grays/silver */
                    --bg:#f5f7fb;           /* very light blue-gray */
                    --ink:#1f2937;          /* slate-800 */
                    --card:#ffffff; 
                    --card-border:#e5e7eb;  /* gray-200 */
                    --muted:#6b7280;        /* gray-500 */
                    --brand:#0ea5e9;        /* PRO cyan */
                    --brand-ink:#ffffff;
                    --field:#f3f4f6; 
                    --field-border:#d1d5db; /* gray-300 */
                    --btn:#0ea5e9; 
                    --btn-ink:#ffffff;
                    --link:#0ea5e9;
                    --shadow:0 12px 40px rgba(15,23,42,.08);
                }
                *{box-sizing:border-box}
                html,body{height:100%}
                body{
                    margin:0; font-family:system-ui,Segoe UI,Roboto,Helvetica,Arial,sans-serif;
                    /* subtle cool vignette */
                    background:
                      radial-gradient(1200px 600px at 70% -10%, rgba(14,165,233,.08), transparent 60%),
                      var(--bg);
                    color:var(--ink); display:flex; align-items:center; justify-content:center;
                }
                .wrap{width:360px; max-width:92vw;}
                .brand{display:flex; align-items:center; gap:10px; justify-content:center; margin-bottom:14px;}
                .badge{background:var(--brand); color:var(--brand-ink); padding:4px 8px; font-size:11px; border-radius:999px; font-weight:700; letter-spacing:.4px;}
                .card{background:var(--card); border:1px solid var(--card-border); border-radius:14px; box-shadow:var(--shadow); padding:22px;}
                h1{margin:0 0 14px; font-size:18px; text-align:center}
                label{display:block; font-size:12px; color:var(--muted); margin:10px 0 6px}
                input{width:100%; padding:10px 12px; border-radius:10px; border:1px solid var(--field-border); background:var(--field); color:var(--ink);}
                .btn{margin-top:14px; width:100%; padding:10px; border-radius:10px; border:0; background:linear-gradient(180deg, var(--btn), #0ea5e9 65%); color:var(--btn-ink); font-weight:700; cursor:pointer;}
                .meta{margin-top:10px; font-size:12px; color:var(--muted); text-align:center}
                a{color:var(--link); text-decoration:none}
                a:hover{text-decoration:underline}
            </style>
        </head>
        <body>
            <div class="wrap">
                <div class="brand">
                    <svg width="28" height="28" viewBox="0 0 24 24" role="img" aria-label="Shield" fill="none">
                        <path d="M12 2l7 3v6c0 5-3.5 9.5-7 11-3.5-1.5-7-6-7-11V5l7-3z" stroke="currentColor" stroke-width="1.2"/>
                        <path d="M9.5 11.5a2.5 2.5 0 115 0v2.2a.8.8 0 01-.8.8H10.3a.8.8 0 01-.8-.8v-2.2z" stroke="currentColor" stroke-width="1.2"/>
                    </svg>
                    <div style="font-weight:800; letter-spacing:.2px">Secure Login Shield</div>
                    <span class="badge">PRO</span>
                </div>
                <div class="card">
                    <h1>Log In</h1>
                    <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
                        <label for="user_login">Username or Email Address</label>
                        <input id="user_login" name="log" type="text" autocomplete="username">
                        <label for="user_pass">Password</label>
                        <input id="user_pass" name="pwd" type="password" autocomplete="current-password">
                        <button class="btn" type="submit">Log In</button>
                        <div class="meta"><a href="#">Lost your password?</a></div>
                    </form>
                </div>
            </div>
        </body>
        </html>
        <?php
    }

    /* ---------------- Admin UI ---------------- */

    public function admin_menu(){
        add_options_page('Secure Login Shield Pro', 'Secure Login Shield Pro', 'manage_options', 'sls-pro', [$this,'settings_page']);
    }

    public function register_settings(){
        register_setting('sls_pro_group', SLS_PRO_OPT, [$this,'sanitize_settings']);
    }

    public function settings_page(){
        if (!current_user_can('manage_options')) return;
        $o = $this->opts;

        // Quick stats
        list($count_total, $count_today, $count_hp, $count_denied) = $this->log_counts();

        // Logo (optional asset)
        $logo_url = plugin_dir_url(__FILE__) . 'assets/sls-pro-logo.png';
        ?>
        <div class="wrap">
            <div style="display:flex;align-items:center;gap:14px;margin:4px 0 8px">
                <img src="<?php echo esc_url($logo_url);?>" alt="Secure Login Shield PRO" style="width:56px;height:56px;border-radius:12px;object-fit:cover;border:1px solid rgba(0,0,0,.06)">
                <h1 style="margin:0">Secure Login Shield <span style="color:#0ea5e9">PRO</span></h1>
            </div>

            <style>
                /* Light-only admin palette (blues + grays/silver) */
                :root {
                  --sls-bg:#f6f8fb; --sls-surface:#ffffff; --sls-border:#e6e8eb;
                  --sls-ink:#212121; --sls-muted:#6f7785;
                  --sls-brand:#0ea5e9; --sls-brand-ink:#ffffff;
                  --sls-accent:#22c55e;
                }
                .sls-tabs{margin-top:12px}
                .nav-tab{cursor:pointer}
                .nav-tab-active{border-bottom-color:var(--sls-brand)!important;color:var(--sls-ink)!important}
                .sls-section{display:none;margin-top:16px}
                .sls-section.active{display:block}
                .sls-grid{display:grid;grid-template-columns:repeat(3,minmax(0,1fr));gap:12px}
                .sls-card{
                  background:var(--sls-surface);
                  border:1px solid var(--sls-border);
                  border-radius:12px; padding:16px;
                  box-shadow:0 6px 20px rgba(0,0,0,.05);
                }
                .sls-kv{display:flex;justify-content:space-between;gap:8px;margin:6px 0}
                .sls-muted{color:var(--sls-muted)}
                textarea.large{width:100%;height:120px}
                input.reg{width:420px;max-width:100%}
                table.widefat td,table.widefat th{vertical-align:top}
                .sls-chip{
                  display:inline-block;padding:2px 8px;border-radius:999px;font-size:12px;
                  background:var(--sls-brand);color:var(--sls-brand-ink);font-weight:600
                }
                @media (max-width: 960px){ .sls-grid{grid-template-columns:1fr} }
            </style>
            <script>
                function slsTab(e,id){
                    e.preventDefault();
                    document.querySelectorAll('.nav-tab').forEach(t=>t.classList.remove('nav-tab-active'));
                    e.target.classList.add('nav-tab-active');
                    document.querySelectorAll('.sls-section').forEach(s=>s.classList.remove('active'));
                    document.getElementById('sls-'+id).classList.add('active');
                    const u=new URL(window.location); u.searchParams.set('tab', id); history.replaceState({},'',u);
                }
                document.addEventListener('DOMContentLoaded',()=>{
                    const tab = new URL(location).searchParams.get('tab') || 'overview';
                    document.getElementById('sls-'+tab)?.classList.add('active');
                    document.querySelector(`.nav-tab[href="#${tab}"]`)?.classList.add('nav-tab-active');
                    if(!document.querySelector('.nav-tab-active')){
                        document.getElementById('sls-overview').classList.add('active');
                        document.querySelector('.nav-tab')?.classList.add('nav-tab-active');
                    }
                });
            </script>

            <h2 class="nav-tab-wrapper sls-tabs">
                <a class="nav-tab" href="#overview" onclick="return slsTab(event,'overview')">Overview</a>
                <a class="nav-tab" href="#secrets" onclick="return slsTab(event,'secrets')">Secrets</a>
                <a class="nav-tab" href="#access" onclick="return slsTab(event,'access')">Access Rules</a>
                <a class="nav-tab" href="#honeypot" onclick="return slsTab(event,'honeypot')">Honeypot</a>
                <a class="nav-tab" href="#logs" onclick="return slsTab(event,'logs')">Logs & Export</a>
                <a class="nav-tab" href="#help" onclick="return slsTab(event,'help')">Help</a>
            </h2>

            <!-- OVERVIEW -->
            <div id="sls-overview" class="sls-section">
                <div class="sls-grid">
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Status</h3>
                        <div class="sls-kv"><span>Honeypot</span><strong><?php echo $o['honeypot_enabled']?'Enabled':'Disabled';?></strong></div>
                        <div class="sls-kv"><span>IP Whitelist</span><strong><?php echo $o['ip_whitelist'] ? 'Set' : '—';?></strong></div>
                        <div class="sls-kv"><span>IP Blacklist</span><strong><?php echo $o['ip_blacklist'] ? 'Set' : '—';?></strong></div>
                        <div class="sls-kv"><span>Schedule</span><strong><?php echo $o['schedule_enabled'] ? esc_html(implode(', ', $o['schedule_days']).' '.$o['schedule_start'].'–'.$o['schedule_end']) : 'Disabled';?></strong></div>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Secret URLs</h3>
                        <div class="sls-kv"><span>Global</span><strong><?php echo count((array)$o['global_secrets']);?></strong></div>
                        <div class="sls-kv"><span>Role-based</span><strong><?php echo count(array_filter($o['role_secrets']??[]));?></strong></div>
                        <div class="sls-kv"><span>Per-user</span><strong><?php echo $this->count_user_secrets();?></strong></div>
                        <p class="sls-muted" style="margin-top:8px">After changing slugs: Settings → Permalinks → Save.</p>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Recent Activity <span class="sls-chip">Live</span></h3>
                        <div class="sls-kv"><span>Total Events</span><strong><?php echo intval($count_total);?></strong></div>
                        <div class="sls-kv"><span>Today</span><strong><?php echo intval($count_today);?></strong></div>
                        <div class="sls-kv"><span>Honeypot Hits</span><strong><?php echo intval($count_hp);?></strong></div>
                        <div class="sls-kv"><span>Access Denied</span><strong><?php echo intval($count_denied);?></strong></div>
                    </div>
                </div>
            </div>

            <!-- BEGIN: SETTINGS FORM (tabs that save options) -->
            <form method="post" action="options.php">
                <?php settings_fields('sls_pro_group'); wp_nonce_field(SLS_PRO_NONCE, SLS_PRO_NONCE); ?>

                <!-- SECRETS -->
                <div id="sls-secrets" class="sls-section">
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Global Secret URLs</h3>
                        <p>One slug per line, e.g. <code>secure-login</code>. Visiting <code>/your-slug</code> loads the real login.</p>
                        <textarea class="large" name="<?php echo SLS_PRO_OPT; ?>[global_secrets]"><?php echo esc_textarea(implode("\n",(array)$o['global_secrets']));?></textarea>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Role-based Secret URLs</h3>
                        <?php foreach(wp_roles()->roles as $role_key=>$role): ?>
                            <p><label><?php echo esc_html($role['name']);?> → <input class="reg" type="text" name="<?php echo SLS_PRO_OPT; ?>[role_secrets][<?php echo esc_attr($role_key);?>]" value="<?php echo esc_attr($o['role_secrets'][$role_key] ?? '');?>"></label></p>
                        <?php endforeach;?>
                        <p class="sls-muted">Users in a role can use the role’s secret URL. Per-user secrets are in each user’s profile.</p>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Post-Login Redirects (per role)</h3>
                        <?php foreach(wp_roles()->roles as $role_key=>$role): ?>
                            <p><label><?php echo esc_html($role['name']);?> → <input class="reg" type="url" name="<?php echo SLS_PRO_OPT; ?>[role_redirects][<?php echo esc_attr($role_key);?>]" value="<?php echo esc_attr($o['role_redirects'][$role_key] ?? '');?>" placeholder="https://example.com/dashboard"></label></p>
                        <?php endforeach;?>
                    </div>
                </div>

                <!-- ACCESS -->
                <div id="sls-access" class="sls-section">
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">IP Controls</h3>
                        <p><strong>Whitelist</strong> (IPv4/IPv6 or CIDR, one per line)</p>
                        <textarea class="large" name="<?php echo SLS_PRO_OPT; ?>[ip_whitelist]" placeholder="1.2.3.4&#10;203.0.113.0/24&#10;::1"><?php echo esc_textarea($o['ip_whitelist']);?></textarea>
                        <p><strong>Blacklist</strong></p>
                        <textarea class="large" name="<?php echo SLS_PRO_OPT; ?>[ip_blacklist]"><?php echo esc_textarea($o['ip_blacklist']);?></textarea>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Country Rules</h3>
                        <p>
                            <select name="<?php echo SLS_PRO_OPT; ?>[country_mode]">
                                <option value="off" <?php selected($o['country_mode'],'off');?>>Off</option>
                                <option value="allow" <?php selected($o['country_mode'],'allow');?>>Allow only these</option>
                                <option value="deny" <?php selected($o['country_mode'],'deny');?>>Block these</option>
                            </select>
                            <input class="reg" type="text" name="<?php echo SLS_PRO_OPT; ?>[country_list]" value="<?php echo esc_attr($o['country_list']);?>" placeholder="US,CA,GB">
                        </p>
                        <p class="sls-muted">Provide country via <code>sls_pro_detect_country</code> filter (use a GeoIP plugin or custom code).</p>
                    </div>
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Schedule (site timezone: <?php echo esc_html(wp_timezone_string());?>)</h3>
                        <p><label><input type="checkbox" name="<?php echo SLS_PRO_OPT; ?>[schedule_enabled]" value="1" <?php checked($o['schedule_enabled'],1);?>> Enable schedule</label></p>
                        <p>
                            <?php foreach(['mon','tue','wed','thu','fri','sat','sun'] as $k=>$lbl): ?>
                                <label style="margin-right:10px"><input type="checkbox" name="<?php echo SLS_PRO_OPT; ?>[schedule_days][]" value="<?php echo $k;?>" <?php checked(in_array($k,(array)$o['schedule_days'],true));?>> <?php echo $lbl;?></label>
                            <?php endforeach; ?>
                        </p>
                        <p>Window:
                            <input type="time" name="<?php echo SLS_PRO_OPT; ?>[schedule_start]" value="<?php echo esc_attr($o['schedule_start']);?>"> –
                            <input type="time" name="<?php echo SLS_PRO_OPT; ?>[schedule_end]" value="<?php echo esc_attr($o['schedule_end']);?>">
                        </p>
                    </div>
                </div>

                <!-- HONEYPOT (RESTORED) -->
                <div id="sls-honeypot" class="sls-section">
                    <div class="sls-card">
                        <h3 style="margin:0 0 8px">Honeypot</h3>
                        <p><label><input type="checkbox" name="<?php echo SLS_PRO_OPT; ?>[honeypot_enabled]" value="1" <?php checked($o['honeypot_enabled'],1);?>> Enable fake <code>/wp-login.php</code></label></p>
                        <p>Auto-block IP for
                            <input class="small-text" type="number" min="1" name="<?php echo SLS_PRO_OPT; ?>[honeypot_block_minutes]" value="<?php echo esc_attr($o['honeypot_block_minutes']);?>"> minutes after a hit.
                        </p>
                        <p class="sls-muted">Tip: add a secret like <code>/secure-login</code> in the Secrets tab. Only requests via a secret URL see the real login.</p>
                    </div>
                </div>

                <?php submit_button('Save Settings'); ?>
            </form>
            <!-- END: SETTINGS FORM -->

            <!-- LOGS (standalone; separate from the POST form to avoid nesting) -->
            <div id="sls-logs" class="sls-section">
                <div class="sls-card">
                    <h3 style="margin:0 0 8px">Logs</h3>
                    <p class="sls-muted" style="margin-top:0">Retention: <?php echo intval($o['log_keep_days']);?> days</p>
                    <?php $this->render_logs_controls_and_table(); ?>
                    <p style="margin-top:12px">
                        <a class="button" href="<?php echo esc_url(admin_url('admin-post.php?action=sls_pro_export&type=csv&_wpnonce='.wp_create_nonce('sls_export')));?>">Export CSV</a>
                        <a class="button" href="<?php echo esc_url(admin_url('admin-post.php?action=sls_pro_export&type=json&_wpnonce='.wp_create_nonce('sls_export')));?>">Export JSON</a>
                    </p>
                </div>
            </div>

            <!-- HELP -->
            <div id="sls-help" class="sls-section">
                <div class="sls-card">
                    <h3 style="margin:0 0 8px">Help</h3>
                    <ul style="margin-top:6px;line-height:1.6">
                        <li>Default secret URL is <code>/secure-login</code> (you can add more in Secrets).</li>
                        <li>After changing secret slugs, visit <strong>Settings → Permalinks</strong> and click “Save” to flush rewrite rules.</li>
                        <li>Per-user secret slugs live in each user’s profile (Admins only can edit).</li>
                        <li>Country allow/deny requires a GeoIP provider to supply ISO-2 via the <code>sls_pro_detect_country</code> filter.</li>
                        <li>Use the Logs tab to filter by event and export CSV/JSON.</li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }

    public function sanitize_settings($in){
        if (!isset($_POST[SLS_PRO_NONCE]) || !wp_verify_nonce($_POST[SLS_PRO_NONCE], SLS_PRO_NONCE)) return $this->opts;

        $out = $this->defaults();

        // Global secrets: textarea -> lines
        $gs = array_filter(array_map('sanitize_title', preg_split('~\R+~', $in['global_secrets'] ?? '')));
        $out['global_secrets'] = array_values(array_unique($gs));

        // Role secrets
        $out['role_secrets'] = [];
        if (!empty($in['role_secrets']) && is_array($in['role_secrets'])){
            foreach($in['role_secrets'] as $k=>$v) $out['role_secrets'][$k] = sanitize_title($v);
        }

        // Role redirects
        $out['role_redirects'] = [];
        if (!empty($in['role_redirects']) && is_array($in['role_redirects'])){
            foreach($in['role_redirects'] as $k=>$v) $out['role_redirects'][$k] = esc_url_raw(trim($v));
        }

        // IP lists
        $out['ip_whitelist'] = $this->normalize_lines($in['ip_whitelist'] ?? '');
        $out['ip_blacklist'] = $this->normalize_lines($in['ip_blacklist'] ?? '');

        // Country rules
        $out['country_mode'] = in_array(($in['country_mode'] ?? 'off'), ['off','allow','deny'], true) ? $in['country_mode'] : 'off';
        $out['country_list'] = strtoupper(preg_replace('/[^A-Z,]/','', $in['country_list'] ?? ''));

        // Schedule
        $out['schedule_enabled'] = !empty($in['schedule_enabled']) ? 1 : 0;
        $days = array_intersect(['mon','tue','wed','thu','fri','sat','sun'], (array)($in['schedule_days'] ?? []));
        $out['schedule_days'] = array_values($days);
        $out['schedule_start'] = preg_match('/^\d{2}:\d{2}$/', $in['schedule_start'] ?? '') ? $in['schedule_start'] : '00:00';
        $out['schedule_end']   = preg_match('/^\d{2}:\d{2}$/', $in['schedule_end']   ?? '') ? $in['schedule_end']   : '23:59';

        // Honeypot
        $out['honeypot_enabled'] = !empty($in['honeypot_enabled']) ? 1 : 0;
        $out['honeypot_block_minutes'] = max(1, intval($in['honeypot_block_minutes'] ?? 30));

        // Logging
        $out['log_enabled'] = !empty($in['log_enabled']) ? 1 : 0; // hidden toggle for compat
        $out['log_keep_days'] = max(1, intval($in['log_keep_days'] ?? 30));

        $this->opts = $out;
        $this->add_rewrite_rules();
        flush_rewrite_rules(false);
        return $out;
    }

    /* ---------------- Logs UI ---------------- */

    private function render_logs_controls_and_table(){
        global $wpdb;
        $table = $wpdb->prefix . SLS_PRO_TABLE;

        // Retention trim
        if (!empty($this->opts['log_keep_days'])){
            $wpdb->query($wpdb->prepare("DELETE FROM $table WHERE occurred_at < (NOW() - INTERVAL %d DAY)", intval($this->opts['log_keep_days'])));
        }

        // Read filters
        $event = sanitize_key($_GET['sls_event'] ?? 'all');
        $q     = trim(sanitize_text_field($_GET['sls_q'] ?? ''));
        $limit = intval($_GET['sls_limit'] ?? 50);
        if (!in_array($limit,[25,50,100],true)) $limit = 50;

        // Build WHERE
        $where = "1=1";
        $args  = [];

        if (in_array($event, ['honeypot-hit','access-denied'], true)){
            $where .= " AND event = %s";
            $args[] = $event;
        }

        // Path contains OR exact IP match
        $ip_bin = null;
        if ($q !== '') {
            if (filter_var($q, FILTER_VALIDATE_IP)) {
                $ip_bin = $this->ip_text_to_bin($q);
            }
            if ($ip_bin !== null) {
                $where .= " AND ip = %s";
                $args[] = $ip_bin;
            } else {
                $where .= " AND (path LIKE %s)";
                $args[] = '%'.$wpdb->esc_like($q).'%';
            }
        }

        // Query
        $sql  = "SELECT id, occurred_at, ip, path, event FROM $table WHERE $where ORDER BY id DESC LIMIT %d";
        $args[] = $limit;
        $rows = $wpdb->get_results($wpdb->prepare($sql, ...$args));

        // Controls (GET form; not nested in POST)
        ?>
        <form method="get" style="margin:0 0 10px">
            <input type="hidden" name="page" value="sls-pro">
            <input type="hidden" name="tab" value="logs">
            <label>Event:
                <select name="sls_event">
                    <option value="all" <?php selected($event,'all');?>>All</option>
                    <option value="honeypot-hit" <?php selected($event,'honeypot-hit');?>>Honeypot hits</option>
                    <option value="access-denied" <?php selected($event,'access-denied');?>>Access denied</option>
                </select>
            </label>
            &nbsp; <label>Search:
                <input type="search" name="sls_q" value="<?php echo esc_attr($q);?>" placeholder="path contains… or exact IP">
            </label>
            &nbsp; <label>Limit:
                <select name="sls_limit">
                    <option <?php selected($limit,25);?>>25</option>
                    <option <?php selected($limit,50);?>>50</option>
                    <option <?php selected($limit,100);?>>100</option>
                </select>
            </label>
            &nbsp; <button class="button">Apply</button>
        </form>
        <?php

        echo '<table class="widefat striped"><thead><tr><th>ID</th><th>Time</th><th>IP</th><th>Path</th><th>Event</th></tr></thead><tbody>';
        if (!$rows){
            echo '<tr><td colspan="5">No events found.</td></tr>';
        } else {
            foreach($rows as $r){
                echo '<tr>';
                echo '<td>'.intval($r->id).'</td>';
                echo '<td>'.esc_html($r->occurred_at).'</td>';
                echo '<td>'.esc_html($this->ip_bin_to_text($r->ip)).'</td>';
                echo '<td>'.esc_html($r->path).'</td>';
                echo '<td>'.esc_html($r->event).'</td>';
                echo '</tr>';
            }
        }
        echo '</tbody></table>';
    }

    public function handle_export(){
        if (!current_user_can('manage_options')) wp_die('Nope');
        check_admin_referer('sls_export');

        global $wpdb;
        $table = $wpdb->prefix . SLS_PRO_TABLE;
        $rows = $wpdb->get_results("SELECT occurred_at, ip, path, ua, ref, event, meta FROM $table ORDER BY id DESC", ARRAY_A);

        $type = sanitize_key($_GET['type'] ?? 'csv');
        if ($type === 'json'){
            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename="sls-logs.json"');
            foreach($rows as &$r){ $r['ip'] = $this->ip_bin_to_text($r['ip']); }
            echo wp_json_encode($rows, JSON_PRETTY_PRINT);
            exit;
        } else {
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="sls-logs.csv"');
            $out = fopen('php://output','w');
            fputcsv($out, array_keys(reset($rows) ?: ['occurred_at','ip','path','ua','ref','event','meta']));
            foreach($rows as $r){
                $r['ip'] = $this->ip_bin_to_text($r['ip']);
                fputcsv($out, $r);
            }
            fclose($out);
            exit;
        }
    }

    /* ---------------- Role-based redirect after login ---------------- */

    public function role_based_redirect($redirect_to, $requested, $user){
        if (is_wp_error($user) || !$user) return $redirect_to;
        $roles = (array)($user->roles ?? []);
        foreach($roles as $r){
            $url = $this->opts['role_redirects'][$r] ?? '';
            if ($url) return $url;
        }
        return $redirect_to;
    }

    /* ---------------- User profile: per-user secret ---------------- */

    public function user_profile_field($user){
        if (!current_user_can('manage_options')) return;
        $val = get_user_meta($user->ID, 'sls_pro_secret_slug', true);
        ?>
        <h2>Secure Login Shield Pro</h2>
        <table class="form-table">
            <tr>
                <th><label for="sls_pro_secret_slug">Personal Secret Login URL</label></th>
                <td>
                    <input type="text" name="sls_pro_secret_slug" id="sls_pro_secret_slug" value="<?php echo esc_attr($val);?>" class="regular-text" placeholder="secure-yourname">
                    <p class="description">Visiting <code>/your-slug</code> will show the real login for this user.</p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function save_user_profile_field($user_id){
        if (!current_user_can('manage_options')) return;
        $val = isset($_POST['sls_pro_secret_slug']) ? sanitize_title($_POST['sls_pro_secret_slug']) : '';
        if ($val) update_user_meta($user_id, 'sls_pro_secret_slug', $val);
        else delete_user_meta($user_id, 'sls_pro_secret_slug');
        $this->add_rewrite_rules();
        flush_rewrite_rules(false);
    }

    /* ---------------- Utilities ---------------- */

    private function log_counts(){
        global $wpdb;
        $t = $wpdb->prefix . SLS_PRO_TABLE;
        $total   = intval($wpdb->get_var("SELECT COUNT(*) FROM $t"));
        $today   = intval($wpdb->get_var("SELECT COUNT(*) FROM $t WHERE DATE(occurred_at)=CURDATE()"));
        $hp      = intval($wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $t WHERE event=%s", 'honeypot-hit')));
        $denied  = intval($wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $t WHERE event=%s", 'access-denied')));
        return [$total, $today, $hp, $denied];
    }

    private function count_user_secrets(){
        $users = get_users(['fields'=>['ID']]);
        $n = 0; foreach($users as $u){ if (get_user_meta($u->ID,'sls_pro_secret_slug',true)) $n++; }
        return $n;
    }

    private function deny_with_403(){
        status_header(403);
        wp_die('<h1>403 Forbidden</h1><p>Access denied.</p>', 'Forbidden', 403);
    }

    private function lines_to_list($s){
        $a = preg_split('~\R+~',$s);
        $a = array_values(array_filter(array_map('trim',$a)));
        return $a;
    }
    private function normalize_lines($s){
        $a = $this->lines_to_list($s);
        return implode("\n",$a);
    }

    private function ip_in_rule($ip, $rule){
        if (strpos($rule,'/') !== false){
            list($subnet, $mask) = explode('/', $rule, 2);
            $mask = intval($mask);
            return $this->ip_in_cidr($ip, $subnet, $mask);
        } else {
            return $ip === $rule;
        }
    }

    private function ip_in_cidr($ip, $subnet, $mask){
        $ip_bin = inet_pton($ip);
        $sub_bin = inet_pton($subnet);
        if ($ip_bin === false || $sub_bin === false) return false;

        $bytes = intdiv($mask, 8);
        $remainder = $mask % 8;

        if (strncmp($ip_bin, $sub_bin, $bytes) !== 0) return false;
        if ($remainder === 0) return true;

        $ip_byte  = ord($ip_bin[$bytes]);
        $sub_byte = ord($sub_bin[$bytes]);
        $mask_byte = (~(255 >> $remainder)) & 255;
        return (($ip_byte & $mask_byte) === ($sub_byte & $mask_byte));
    }

    private function client_ip(){
        foreach (['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_X_REAL_IP','REMOTE_ADDR'] as $k){
            if (!empty($_SERVER[$k])){
                $iplist = explode(',', $_SERVER[$k]);
                $ip = trim($iplist[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
            }
        }
        return null;
    }

    private function ip_text_to_bin($ip){
        $p = @inet_pton($ip);
        return $p === false ? null : $p;
    }
    private function ip_bin_to_text($bin){
        if ($bin === null) return '';
        return @inet_ntop($bin) ?: '';
    }

    private function req_path(){
        return parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
    }

    private function user_agent(){
        return substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 1000);
    }
    private function referrer(){
        return substr($_SERVER['HTTP_REFERER'] ?? '', 0, 1000);
    }

    private function log_event($event, $meta = []){
        if (empty($this->opts['log_enabled'])) return;
        global $wpdb;
        $table = $wpdb->prefix . SLS_PRO_TABLE;
        $ip = $this->client_ip();
        $wpdb->insert($table, [
            'occurred_at' => current_time('mysql', 1),
            'ip'          => $this->ip_text_to_bin($ip) ?: str_repeat("\0", 16),
            'path'        => substr($this->req_path(), 0, 255),
            'ua'          => $this->user_agent(),
            'ref'         => $this->referrer(),
            'event'       => sanitize_key($event),
            'meta'        => wp_json_encode($meta),
        ], ['%s','%s','%s','%s','%s','%s','%s']);
    }
}

SLS_Pro::instance();
